/*
 * Copyright (c) 2024 FabulaTech.
 * All rights reserved.
 * Permission to use this software as a part of FabulaTech solution only is
 * granted.
 * http://www.fabulatech.com
 */

#include <linux/slab.h>
#include <linux/list.h>
#include <linux/spinlock.h>
#include "ftdefs.h"
#include "fthclink.h"

/* Unique id given to urbs */
static unsigned int seqnum = 0;

/* List of urbs, enqueued, but not send to user-mode */
static LIST_HEAD(pending);

/* List of urbs submitted to the endpoint and still active */
static LIST_HEAD(submitted);

static spinlock_t priv_lock;

/* Internal urb representation */
struct urbpriv {
	struct semaphore sem;
	/* Defined in header file as FUNC_... */
	unsigned int func;
	/* Kernel urb */
	struct urb *urb;
	/* Port number (also used for RESET request) */
	int port;
	/* Sequence number */
	unsigned int seq;
	/* We need this pipe after unlinking */
	unsigned int pipe;
	/* Used to free urb automatically */
	unsigned int refcount;
	/* Used to store urb in the list of submitted urbs */
	struct list_head slist;
	/* Used to store urb in the list of pending urbs */
	struct list_head plist;
};

struct urbpriv *urbpriv_alloc(gfp_t mem_flags) {
	struct urbpriv *urbpriv;
	urbpriv = kmalloc(sizeof(struct urbpriv), mem_flags);
	if (urbpriv == NULL)
		return NULL;
	urbpriv->refcount = 1;
	return urbpriv;
}

void urbpriv_init(struct urbpriv *urbpriv, struct urb *urb,
    unsigned int func, int port) {
	unsigned long flags;
	sema_init(&urbpriv->sem, 1);
	urbpriv->func = func;
	urbpriv->urb = urb;
	urbpriv->port = port;
	INIT_LIST_HEAD(&urbpriv->slist);
	INIT_LIST_HEAD(&urbpriv->plist);
	spin_lock_irqsave(&priv_lock, flags);
	urbpriv->seq = seqnum++;
	if (urb != NULL) {
		urbpriv->pipe = urb->pipe;
		urb->hcpriv = urbpriv;
	} else
		urbpriv->pipe = 0;
	spin_unlock_irqrestore(&priv_lock, flags);
}

static void urbpriv_release(struct urbpriv *urbpriv) {
	kfree(urbpriv);
}

int urbpriv_lock(struct urbpriv *urbpriv) {
	return down_interruptible(&urbpriv->sem);
}

void urbpriv_unlock(struct urbpriv *urbpriv) {
	up(&urbpriv->sem);
}

struct urb *urbpriv_urb(struct urbpriv *urbpriv) {
	return urbpriv->urb;
}

unsigned int urbpriv_func(struct urbpriv *urbpriv) {
	return urbpriv->func;
}

unsigned int urbpriv_seq(struct urbpriv *urbpriv) {
	return urbpriv->seq;
}

unsigned int urbpriv_pipe(struct urbpriv *urbpriv) {
	return urbpriv->pipe;
}

int urbpriv_port(struct urbpriv *urbpriv) {
	return urbpriv->port;
}

void urbpriv_set_func(struct urbpriv *urbpriv, unsigned int func) {
	urbpriv->func = func;
}

void urbpriv_unlink(struct urbpriv *urbpriv) {
	unsigned long flags;
	spin_lock_irqsave(&priv_lock, flags);
	if (urbpriv->urb != NULL) {
		urbpriv->urb->hcpriv = NULL;
		urbpriv->urb = NULL;
	}
	spin_unlock_irqrestore(&priv_lock, flags);
}

void urbpriv_unref(struct urbpriv *urbpriv) {
	unsigned long flags;
	unsigned int refcount;
	spin_lock_irqsave(&priv_lock, flags);
	refcount = --(urbpriv->refcount);
	spin_unlock_irqrestore(&priv_lock, flags);
	if (refcount == 0)
		urbpriv_release(urbpriv);
}

void urbpriv_unlock_unref(struct urbpriv *urbpriv) {
	unsigned long flags;
	unsigned int refcount;
	up(&urbpriv->sem);
	spin_lock_irqsave(&priv_lock, flags);
	refcount = --(urbpriv->refcount);
	spin_unlock_irqrestore(&priv_lock, flags);
	if (refcount == 0)
		urbpriv_release(urbpriv);
}

struct urbpriv *urb_to_urbpriv(struct urb *urb) {
	struct urbpriv *urbpriv;
	unsigned long flags;
	spin_lock_irqsave(&priv_lock, flags);
	urbpriv = (struct urbpriv *) urb->hcpriv;
	if (urbpriv != NULL)
		urbpriv->refcount++;
	spin_unlock_irqrestore(&priv_lock, flags);
	return urbpriv;
}

struct urbpriv *urbpriv_get_pending_urb() {
	struct urbpriv *urbpriv;
	unsigned long flags;
	spin_lock_irqsave(&priv_lock, flags);
	if (!list_empty(&pending)) {
		urbpriv = list_first_entry(&pending, struct urbpriv, plist);
		urbpriv->refcount++;
	} else
		urbpriv = NULL;
	spin_unlock_irqrestore(&priv_lock, flags);
	return urbpriv;
}

struct urbpriv *urbpriv_get_submitted_urb_reverse() {
	struct urbpriv *urbpriv;
	unsigned long flags;
	spin_lock_irqsave(&priv_lock, flags);
	if (!list_empty(&submitted)) {
		urbpriv = list_last_entry(&submitted, struct urbpriv, slist);
		urbpriv->refcount++;
	} else
		urbpriv = NULL;
	spin_unlock_irqrestore(&priv_lock, flags);
	return urbpriv;
}

struct urbpriv *urbpriv_by_seq(unsigned long seq) {
	struct urbpriv *urbpriv;
	unsigned long flags;
	spin_lock_irqsave(&priv_lock, flags);
	list_for_each_entry(urbpriv, &submitted, slist) {
		if (urbpriv->seq == seq) {
			urbpriv->refcount++;
			spin_unlock_irqrestore(&priv_lock, flags);
			return urbpriv;
		}
	}
	spin_unlock_irqrestore(&priv_lock, flags);
	return NULL;
}

void urbpriv_set_submitted(struct urbpriv *urbpriv) {
	unsigned long flags;
	spin_lock_irqsave(&priv_lock, flags);
	urbpriv->refcount++;
	list_add_tail(&urbpriv->slist, &submitted);
	spin_unlock_irqrestore(&priv_lock, flags);
}

void urbpriv_set_pending(struct urbpriv *urbpriv) {
	unsigned long flags;
	spin_lock_irqsave(&priv_lock, flags);
	urbpriv->refcount++;
	list_add_tail(&urbpriv->plist, &pending);
	spin_unlock_irqrestore(&priv_lock, flags);
}

void urbpriv_unset_submitted(struct urbpriv *urbpriv) {
	unsigned long flags;
	spin_lock_irqsave(&priv_lock, flags);
	list_del(&urbpriv->slist);
	if (--(urbpriv->refcount) == 0) {
		spin_unlock_irqrestore(&priv_lock, flags);
		urbpriv_release(urbpriv);
	} else {
		INIT_LIST_HEAD(&urbpriv->slist);
		spin_unlock_irqrestore(&priv_lock, flags);
	}
}

void urbpriv_unset_pending(struct urbpriv *urbpriv) {
	unsigned long flags;
	spin_lock_irqsave(&priv_lock, flags);
	list_del(&urbpriv->plist);
	if (--(urbpriv->refcount) == 0) {
		spin_unlock_irqrestore(&priv_lock, flags);
		urbpriv_release(urbpriv);
	} else {
		INIT_LIST_HEAD(&urbpriv->plist);
		spin_unlock_irqrestore(&priv_lock, flags);
	}
}

int urbpriv_pending(struct urbpriv *urbpriv) {
	int empty;
	unsigned long flags;
	spin_lock_irqsave(&priv_lock, flags);
	empty = list_empty(&urbpriv->plist);
	spin_unlock_irqrestore(&priv_lock, flags);
	return !empty;
}

int urbpriv_has_pending_urbs() {
	int empty;
	unsigned long flags;
	spin_lock_irqsave(&priv_lock, flags);
	empty = list_empty(&pending);
	spin_unlock_irqrestore(&priv_lock, flags);
	return !empty;
}

void urbpriv_list_init(void) {
	spin_lock_init(&priv_lock);
}
