/*
 * Copyright (c) 2024 FabulaTech.
 * All rights reserved.
 * Permission to use this software as a part of FabulaTech solution only is
 * granted.
 * http://www.fabulatech.com
 */

#include <linux/module.h>

#include <linux/platform_device.h>
#include <linux/usb.h>
#include <linux/device.h>

#include "ftrelease.h"
#include "ftdefs.h"

MODULE_LICENSE(FT_LICENSE);
MODULE_AUTHOR(FT_AUTHOR);

/**
 * Kernel module parameter: verbose
 */
int verbose = 0;
module_param(verbose, int, 0644);
MODULE_PARM_DESC(verbose, "be verbose (default: on)");

static int fthc_refcnt = 0;
static struct urbpriv *spriv[FTHC_MAX_PORTS];
static struct urbpriv *rpriv[FTHC_MAX_PORTS];

int fthc_refcnt_inc(void) {
	unsigned long flags;
	int rc = 0;
	spin_lock_irqsave(&hcd_instance->lock, flags);
	if (fthc_refcnt > 0)
		rc = -EMLINK;
	if (++fthc_refcnt == 1)
		try_module_get(THIS_MODULE);
	spin_unlock_irqrestore(&hcd_instance->lock, flags);
	return rc;
}

void fthc_refcnt_dec(void) {
	unsigned long flags;
	spin_lock_irqsave(&hcd_instance->lock, flags);
	if (fthc_refcnt > 0) {
		if (--fthc_refcnt == 0)
			module_put(THIS_MODULE);
	} else
		printk(KERN_ERR "fthc got negative reference count\n");
	spin_unlock_irqrestore(&hcd_instance->lock, flags);
}

/**
 * fthc_probe: Creates a virtual host controller device
 * and adds it to usb subsystem
 *
 * fthc_remove: Destroys virtual hcd
 */
static int fthc_probe(struct platform_device *pdev) {
	int i, rc;

	logdk("fthc probe\n");
	memset(hcd_instance, 0, sizeof(*hcd_instance));
	spin_lock_init(&hcd_instance->lock);
	for (i = 0; i < FTHC_MAX_PORTS; i++) {
		spriv[i] = urbpriv_alloc(GFP_KERNEL);
		rpriv[i] = urbpriv_alloc(GFP_KERNEL);
		if (spriv[i] == NULL || rpriv[i] == NULL) {
			do {
				if (spriv[i] != NULL)
					urbpriv_unref(spriv[i]);
				if (rpriv[i] != NULL)
					urbpriv_unref(rpriv[i]);

			} while (i-- != 0);
			return -ENOMEM;
		}
	}

	/* Create new hcd */
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,24)
	hcd_instance->hcd_hs = usb_create_hcd(&fthc_driver, &pdev->dev, dev_name(&pdev->dev));
#else
	hcd_instance->hcd_hs = usb_create_hcd(&fthc_driver, &pdev->dev, FT_DRIVER_NAME);
#endif
	if (!hcd_instance->hcd_hs) {
		printk(KERN_ERR "Failed to create hcd\n");
		return -ENOMEM;
	}

	/* Store hcd instance */
	for (i = 0; i < FTHC_MAX_PORTS; i++)
		memset(&hcd_instance->ports[i], 0, sizeof(hcd_instance->ports[0]));

#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,4,0)
#define FTHC_IRQ_NUM 0
#else
#define FTHC_IRQ_NUM -1
#endif
	rc = usb_add_hcd(hcd_instance->hcd_hs, FTHC_IRQ_NUM, 0);
	if (rc) {
		printk(KERN_ERR "Failed to add new hcd\n");
		usb_put_hcd(hcd_instance->hcd_hs);
		hcd_instance->hcd_hs = NULL;
		return rc;
	}

#ifdef USB_DT_SS_HUB
	/* Super speed */
	hcd_instance->hcd_ss = usb_create_shared_hcd(&fthc_driver, &pdev->dev,
			dev_name(&pdev->dev), hcd_instance->hcd_hs);
	if (!hcd_instance->hcd_ss) {
		printk(KERN_ERR "Failed to create shared hcd\n");
		usb_remove_hcd(hcd_instance->hcd_hs);//TODO: use only high speed HCD
		hcd_instance->hcd_hs = NULL;
		return -ENOMEM;
	}
	rc = usb_add_hcd(hcd_instance->hcd_ss, FTHC_IRQ_NUM, 0);
	if (rc) {
		printk(KERN_ERR "Failed to add new ss hcd\n");
		usb_put_hcd(hcd_instance->hcd_ss);
		hcd_instance->hcd_ss = NULL;
		usb_remove_hcd(hcd_instance->hcd_hs);//TODO: use only high speed HCD
		hcd_instance->hcd_hs = NULL;
		return rc;
	}
#endif

	return 0;
}

#if LINUX_VERSION_CODE < KERNEL_VERSION(6,11,0)
static int fthc_remove(struct platform_device *pdev) {
#else
static void fthc_remove(struct platform_device *pdev) {
#endif
	int i;

	logdk("fthc remove\n");
#ifdef USB_DT_SS_HUB
	usb_remove_hcd(hcd_instance->hcd_ss);
	usb_put_hcd(hcd_instance->hcd_ss);
#endif
	usb_remove_hcd(hcd_instance->hcd_hs);
	usb_put_hcd(hcd_instance->hcd_hs);

	hcd_instance->hcd_hs = NULL;
	hcd_instance->hcd_ss = NULL;
	for (i = 0; i < FTHC_MAX_PORTS; i++) {
		urbpriv_unref(spriv[i]);
		urbpriv_unref(rpriv[i]);
	}

#if LINUX_VERSION_CODE < KERNEL_VERSION(6,11,0)
	return 0;
#endif
}

#ifdef CONFIG_PM
static int fthc_suspend(struct platform_device *pdev, pm_message_t state)
{
	struct usb_hcd *hcd;
	unsigned long flags;
	int i;

	logdk("fthc suspend\n");
	hcd = platform_get_drvdata(pdev);

	spin_lock_irqsave(&hcd_instance->lock, flags);
	for (i = 0; i < FTHC_MAX_PORTS; i++)
		if ((hcd_instance->ports[i].status & USB_PORT_STAT_CONNECTION) != 0) {
			hcd_instance->ports[i].status &= ~(USB_PORT_STAT_CONNECTION | USB_SPEED_HIGH | USB_SPEED_LOW);
			hcd_instance->ports[i].status |= (1 << USB_PORT_FEAT_C_CONNECTION);
			hcd_instance->ports[i].speed = 0;
			hcd_instance->ports[i].usbdev = NULL;
			hcd_instance->ports[i].pending = 0;
			hcd_instance->ports[i].suspended = 1;
			link_urb_special(spriv[i], FUNC_SUSPEND, i);
		}
	clear_bit(HCD_FLAG_HW_ACCESSIBLE, &hcd->flags);
	spin_unlock_irqrestore(&hcd_instance->lock, flags);

	return 0;
}

static int fthc_resume(struct platform_device *pdev)
{
	struct usb_hcd *hcd;
	unsigned long flags;
	int i;

	logdk("fthc resume\n");
	hcd = platform_get_drvdata(pdev);

	spin_lock_irqsave(&hcd_instance->lock, flags);
	set_bit(HCD_FLAG_HW_ACCESSIBLE, &hcd->flags);
	for (i = 0; i < FTHC_MAX_PORTS; i++)
		if (hcd_instance->ports[i].suspended) {
			hcd_instance->ports[i].suspended = 0;
			link_urb_special(rpriv[i], FUNC_RESUME, i);
		}
	spin_unlock_irqrestore(&hcd_instance->lock, flags);

	usb_hcd_poll_rh_status(hcd);

	return 0;
}
#else
#define fthc_suspend	NULL
#define fthc_resume		NULL
#endif

/**
 * Virtual HCD is a platform device
 */

static struct platform_driver hcd_driver = {
	.probe = fthc_probe,
	.remove = fthc_remove,
	.suspend = fthc_suspend,
	.resume	= fthc_resume,
	.driver = {
		.name = (char *) FT_DRIVER_NAME,
		.owner = THIS_MODULE,
	},
};

/* Nothing to do on release */
static void fthc_release(struct device *dev) {}

static struct platform_device hcd_device = {
	.name = (char *) FT_DRIVER_NAME,
	.id = -1,
	.dev = {
		.release = fthc_release,
	},
};


/**
 * Generic module initialization/cleanup routines
 *
 * Here a platform driver is loaded and corresponding device (hcd) is created
 */

static int __init init(void) {
	int rc;
	if (usb_disabled()) {
		printk(KERN_ERR "usb subsystem disabled.\n");
		return -ENODEV;
	}

	rc = platform_driver_register(&hcd_driver);
	if (rc < 0) {
		printk(KERN_ERR "failed to register HCD platform driver\n");
		return rc;
	}

	rc = platform_device_register(&hcd_device);
	if (rc < 0) {
		printk(KERN_ERR "failed to create platform device (hcd)\n");
		return rc;
	}

	printk(KERN_INFO "%s module loaded\n", FT_DRIVER_DESCRIPTION);
	return 0;
}

static void __exit cleanup(void) {
	platform_device_unregister(&hcd_device);
	platform_driver_unregister(&hcd_driver);
	printk(KERN_INFO "%s module unloaded\n", FT_DRIVER_DESCRIPTION);
}

module_init(init);
module_exit(cleanup);

