#!/bin/sh

shortname=`basename "$0"`
scriptname="$0"
rootdir=`dirname "$scriptname"`
rootdir=`dirname "$rootdir"`

PROD=ftvusbrdp
DRV=fthc
ICONNAME=fabulatech-usbrdpserver

build_kernel_module()
{
	make -C "$rootdir/module"
	if [ $? -eq 0 ]; then
		return 0
	fi
	echo "Cannot build kernel module" >&2
	return 1
}

remove_kernel_module()
{
	lsmod |grep "$DRV"
	if [ $? -eq 0 ]; then
		rmmod "$DRV"
	fi
	make -C "$rootdir/module" clean
}

is_systemd_os()
{
	systemctl > /dev/null 2>&1
	if [ $? -eq 0 ]; then
		return 0
	else
		return 1
	fi
}

install_service()
{
	is_systemd_os
	if [ $? -eq 0 ]; then
		cp "$rootdir/doc/$PROD.service" /etc/systemd/system &&
			systemctl daemon-reload &&
			systemctl enable $PROD &&
			systemctl start $PROD
		if [ $? -ne 0 ]; then
			echo "Install and run service"
			return 5
		fi
		return 0
	fi

	# for non-systemd OS just return OK
	return 0
}

uninstall_service()
{
	is_systemd_os
	if [ $? -eq 0 ]; then
		systemctl stop $PROD
		systemctl disable $PROD
		rm /etc/systemd/system/$PROD.service
		systemctl daemon-reload
		return 0
	fi

	# for non-systemd OS just return OK
	return 0
}

install_icons()
{
	sizes="16 22 24 32 48 64 128 256"
	for sz in $sizes ; do
		xdg-icon-resource install --size $sz "$rootdir"/share/icons/"$sz"x"$sz"/$ICONNAME.png
	done

	desktop-file-install "$rootdir/share/$ICONNAME.desktop"
}

uninstall_icons()
{
	rm -f /usr/share/applications/$ICONNAME.desktop
	update-desktop-database /usr/share/applications

	sizes="16 22 24 32 48 64 128 256"
	for sz in $sizes ; do
		xdg-icon-resource uninstall --size $sz "$rootdir"/share/icons/"$sz"x"$sz"/$ICONNAME
	done
}

do_install()
{
	check_dep
	if [ $? -ne 0 ]; then
		return 1
	fi

	build_kernel_module && install_service
	if [ $? -eq 0 ]; then
		install_icons
		echo "Installed successfully"
		return 0
	else
		echo "Error occured during installation"
		return 1
	fi
}

do_uninstall()
{
	uninstall_icons > /dev/null 2>&1
	uninstall_service > /dev/null 2>&1
	remove_kernel_module > /dev/null 2>&1
	echo "Uninstalled successfully"
}

check_dep()
{
	echo -n "Checking dependencies: "
	exitcode=0
	make --version > /dev/null 2>&1
	if [ $? -eq 127 ]; then
		echo -n "make "
		exitcode=1
	fi
	gcc --version > /dev/null 2>&1
	if [ $? -eq 127 ]; then
		echo -n "gcc "
		exitcode=1
	fi

	if [ $exitcode -eq 0 ]; then
		echo "ok"
	else
		echo "missing"
	fi
	return $exitcode
}

whoami=`id -u`
if [ "$whoami" = "" ]; then
	echo "cannot determine user" >&2
	exit 1
fi
if [ $whoami -ne 0 ]; then
	echo "Run this script under root user" >&2
	exit 1
fi

TEMP=$(getopt -o 'iu' --long 'install,uninstall' -n "$shortname" -- "$@")

if [ $? -ne 0 ]; then
	echo 'Invalid arguments' >&2
	exit 1
fi

eval set -- "$TEMP"
unset TEMP

iflag=0
uflag=0

while true; do
	case "$1" in
		'-i'|'--install')
			iflag=1
			shift
			continue
		;;
		'-u'|'--uninstall')
			uflag=1
			shift
			continue
		;;
		'--')
			shift
			break
		;;
		*)
			echo 'Invalid argument' >&2
			exit 1
		;;
	esac
done

if [ $iflag -eq 0 -a $uflag -eq 0 -o $iflag -eq 1 -a $uflag -eq 1 ]; then
	echo "One of install or uninstall should be specified" >&2
	exit 1
fi

if [ $iflag -eq 1 ]; then
	do_install
else
	do_uninstall
fi
exit $?
