/*
 * Copyright (c) 2024 FabulaTech.
 * All rights reserved.
 * Permission to use this software as a part of FabulaTech solution only is
 * granted.
 * http://www.fabulatech.com
 */

#include <linux/fs.h>
#include <linux/poll.h>
#include <linux/miscdevice.h>
#include <linux/delay.h>
#include <linux/scatterlist.h>

#include "ftdefs.h"
#include "fthclink.h"

DECLARE_WAIT_QUEUE_HEAD(link_poll_queue);

static inline void show_setup(char *buf, struct urb *urb) {
	struct usb_ctrlrequest *usbctlreq;
	usbctlreq = (struct usb_ctrlrequest *) urb->setup_packet;
	sprintf(buf, "%02x %02x %04x %04x %04x",
			usbctlreq->bRequestType,
			usbctlreq->bRequest,
			usbctlreq->wValue,
			usbctlreq->wIndex,
			usbctlreq->wLength);
}

static inline void show_urb(const char *prefix, struct urb *urb) {
	if (verbose) {
		int ep;
		ep = usb_pipeendpoint(urb->pipe);
		printk(KERN_DEBUG "%s: urb/%p %s ep%d%s%s len %d/%d flag %x (%s) %s\n",
				prefix,
				urb,
				({ char *s;
				 switch(urb->dev->speed) {
				 case USB_SPEED_LOW: s="ls"; break;
				 case USB_SPEED_FULL: s="fs"; break;
				 case USB_SPEED_HIGH: s="hs"; break;
				 case USB_SPEED_SUPER: s="ss"; break;
				 default: s = "ss"; break;
				 }; s; }),
				ep, ep ? (usb_pipein(urb->pipe) ? "in" : "out") : "",
				({ char *s;
				 switch (usb_pipetype(urb->pipe)) {
				 case PIPE_CONTROL: s=""; break;
				 case PIPE_BULK: s="-bulk"; break;
				 case PIPE_INTERRUPT: s="-iso"; break;
				 default: s="-iso"; break;
				 }; s;}),
				urb->actual_length, urb->transfer_buffer_length,
				urb->transfer_flags,
				(urb->status != -EINPROGRESS ? "unlinked" : "linked"),
				usb_pipetype(urb->pipe) == PIPE_CONTROL ? ({ char s[32];
						show_setup(s, urb); s;	 }) : "");
	}
}

void link_urb_special(struct urbpriv *urbpriv, unsigned int func, int port) {
	urbpriv_init(urbpriv, NULL, func, port);
	urbpriv_set_pending(urbpriv);
	wake_up_interruptible(&link_poll_queue);
}

void link_urb_enqueue(struct urb *urb, struct urbpriv *urbpriv, int port) {
	show_urb("enq", urb);
	urbpriv_init(urbpriv, urb, FUNC_URB, port);
	urbpriv_set_pending(urbpriv);
	wake_up_interruptible(&link_poll_queue);
}

int link_urb_dequeue(struct urb *urb) {
	struct urbpriv *urbpriv;

	show_urb("deq", urb);

	/* Get and lock urbpriv */
	urbpriv = urb_to_urbpriv(urb);
	if (urbpriv == NULL)
		return -EIDRM;
	if (urbpriv_lock(urbpriv)) {
		urbpriv_unref(urbpriv);
		return -ERESTARTSYS;
	}

	if (urbpriv_urb(urbpriv) == NULL) { /* Givebacked in ioctl routine */
		urbpriv_unlock_unref(urbpriv);
		return -EIDRM;
	}

	/* Unlink urb from urbpriv */
	urbpriv_unlink(urbpriv);

	/* If still pending - remove from pending list
	 * Otherwise: add pending urb for cancel action */
	if (urbpriv_pending(urbpriv)) {
		urbpriv_unset_pending(urbpriv);
	} else {
		urbpriv_set_func(urbpriv, FUNC_CANCEL);
		urbpriv_unset_submitted(urbpriv);
		urbpriv_set_pending(urbpriv);
	}

	urbpriv_unlock_unref(urbpriv);

	wake_up_interruptible(&link_poll_queue);
	return 0;
}

static unsigned int link_poll(struct file *f, struct poll_table_struct *wait) {
	int pending;

	poll_wait(f, &link_poll_queue, wait);
	pending = urbpriv_has_pending_urbs();
	if (pending)
		return POLLIN | POLLRDNORM;
	else
		return 0;
}

#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 35)
static int link_ioctl(struct inode *node, struct file *f, 
		unsigned int ioctl_num, unsigned long ioctl_param)
#else
static long link_ioctl(struct file *f, 
		unsigned int ioctl_num, unsigned long ioctl_param)
#endif
{

	void *arg = (void *)ioctl_param;
	struct usb_hcd *hcd;
	struct urb *urb;
	struct scatterlist *sg;
	struct urbpriv *urbpriv;
	hcd_port_t port;
	uint32_t size, seq, offset, sz;
	urb_t urbhdr;
	iso_frame_t iso, *piso;
	hcd_dev_addr_t hda;
	unsigned long flags;
	int i, err, status;

	switch (ioctl_num) {
	case IOCTL_PORT_ENABLE:
		logdk("ioctl: port enable\n");

		if (copy_from_user(&port, arg, sizeof(hcd_port_t)))
			return -EFAULT;
		err = fthc_port_enable(&port);
		if (err)
			return err;
		if (copy_to_user(arg, &port, sizeof(hcd_port_t))) {
			fthc_port_disable(&port);
			return -EFAULT;
		}
		hcd = port.port_number < FTHC_HS_MAX_PORTS ? hcd_instance->hcd_hs : hcd_instance->hcd_ss;
		usb_hcd_poll_rh_status(hcd);
		return 0;

	case IOCTL_PORT_DISABLE:
		logdk("ioctl: port disable\n");

		if (copy_from_user(&port, arg, sizeof(hcd_port_t)))
			return -EFAULT;
		fthc_port_disable(&port);
		hcd = port.port_number < FTHC_HS_MAX_PORTS ? hcd_instance->hcd_hs : hcd_instance->hcd_ss;
		usb_hcd_poll_rh_status(hcd);
		return 0;

	case IOCTL_GET_URB_HDR:
		logdk("ioctl: get urb header\n");

		urbpriv = urbpriv_get_pending_urb();
		if (urbpriv == NULL)
			return -EINVAL;
		if (urbpriv_lock(urbpriv)) {
			urbpriv_unref(urbpriv);
			return -ERESTARTSYS;
		}

		urb = urbpriv_urb(urbpriv);
		if (urb == NULL && urbpriv_func(urbpriv) == FUNC_URB) { /* urb dequeued */
			urbpriv_unlock_unref(urbpriv);
			return -ETIMEDOUT;
		}

		urbhdr.size = sizeof(urb_t);
		urbhdr.seq = urbpriv_seq(urbpriv);
		urbhdr.func = urbpriv_func(urbpriv);
		urbhdr.port = urbpriv_port(urbpriv);
		urbhdr.pipe = urbpriv_pipe(urbpriv);
		if (urbhdr.func == FUNC_URB) {
			urbhdr.flags = urb->transfer_flags;
			urbhdr.status = urb->status;
			urbhdr.length = urb->transfer_buffer_length;
			urbhdr.numiso = urb->number_of_packets;
			urbpriv_set_submitted(urbpriv);
		}

		urbpriv_unset_pending(urbpriv);
		urbpriv_unlock_unref(urbpriv);

		if (copy_to_user(arg, &urbhdr, sizeof(urb_t)))
			return -EFAULT;
		return 0;

	case IOCTL_GET_URB_BUF:
		logdk("ioctl: get urb buffer\n");

		if (get_user(size, (uint32_t __user *)&((urb_t *)arg)->size)
		    || get_user(seq, (uint32_t __user *)&((urb_t *)arg)->seq))
			return -EFAULT;

		urbpriv = urbpriv_get_submitted_urb_reverse();
		if (urbpriv == NULL) /* urb dequeued */
			return -ETIMEDOUT;
		if (urbpriv_lock(urbpriv)) {
			urbpriv_unref(urbpriv);
			return -ERESTARTSYS;
		}

		urb = urbpriv_urb(urbpriv);
		if (urb == NULL || urbpriv_seq(urbpriv) != seq) { /* urb dequeued */
			urbpriv_unlock_unref(urbpriv);
			return -ETIMEDOUT;
		}

		/* Check size */
		if (usb_pipeisoc(urb->pipe))
			offset = sizeof(iso_frame_t) * urb->number_of_packets;
		else if (usb_pipecontrol(urb->pipe))
			offset = 8;
		else
			offset = 0;
		if (size < sizeof(urb_t) + offset) {
			urbpriv_unlock_unref(urbpriv);
			return -ERANGE;
		}

		/* Copy urb buffer */
		if (usb_pipeisoc(urb->pipe)) {
			piso = (iso_frame_t *)((urb_t *)arg)->buffer;
			for (i = 0; i < urb->number_of_packets; i++, piso++) {
				if (put_user(urb->iso_frame_desc[i].offset, (uint32_t __user *)&piso->offset)
				    || put_user(urb->iso_frame_desc[i].length, (uint32_t __user *)&piso->length)
				    || put_user(urb->iso_frame_desc[i].status, (int32_t __user *)&piso->status)) {
					urbpriv_unlock_unref(urbpriv);
					return -EFAULT;
				}
			}
		} else if (usb_pipecontrol(urb->pipe) && copy_to_user(((urb_t *)arg)->buffer, urb->setup_packet, 8)) {
			urbpriv_unlock_unref(urbpriv);
			return -EFAULT;
		}
		size -= sizeof(urb_t) + offset;
		if (size > urb->transfer_buffer_length)
			size = urb->transfer_buffer_length;
		if (size) {
			if (urb->num_sgs) {
				for_each_sg(urb->sg, sg, urb->num_sgs, i) {
					sz = min(size, sg->length);
					if (copy_to_user(((urb_t *)arg)->buffer + offset, sg_virt(sg), sz)) {
						urbpriv_unlock_unref(urbpriv);
						return -EFAULT;
					}
					offset += sz;
					size -= sz;
					if (!size)
						break;
				}
			} else {
				if (copy_to_user(((urb_t *)arg)->buffer + offset, urb->transfer_buffer, size)) {
					urbpriv_unlock_unref(urbpriv);
					return -EFAULT;
				}
				offset += size;
			}
		}
		size = sizeof(urb_t) + offset;
		if (put_user(size, (uint32_t __user *)&((urb_t *)arg)->size)) {
			urbpriv_unlock_unref(urbpriv);
			return -EFAULT;
		}

		urbpriv_unlock_unref(urbpriv);
		return 0;

	case IOCTL_COMPLETE_URB:
		logdk("ioctl: complete urb\n");

		if (copy_from_user(&urbhdr, arg, sizeof(urb_t)))
			return -EFAULT;
		if (urbhdr.func != FUNC_URB)
			return 0;

		urbpriv = urbpriv_by_seq(urbhdr.seq);
		if (urbpriv == NULL) /* Canceled */
			return 0;
		if (urbpriv_lock(urbpriv)) {
			urbpriv_unref(urbpriv);
			return -ERESTARTSYS;
		}
		urb = urbpriv_urb(urbpriv);
		if (urb == NULL) { /* Also canceled */
			urbpriv_unlock_unref(urbpriv);
			return 0;
		}

		/* Check size */
		if (usb_pipeisoc(urb->pipe))
			offset = sizeof(iso_frame_t) * urbhdr.numiso;
		else
			offset = 0;
		if (urbhdr.size < sizeof(urb_t) + offset ||
		    urbhdr.length > urb->transfer_buffer_length ||
		    (usb_pipeisoc(urb->pipe) && urb->number_of_packets != urbhdr.numiso)) {
			urbpriv_unlock_unref(urbpriv);
			return -ERANGE;
		}

		/* Copy urb */
		spin_lock_irqsave(&hcd_instance->lock, flags);
		status = urb->status = urbhdr.status;
		urb->actual_length = urbhdr.length;
		spin_unlock_irqrestore(&hcd_instance->lock, flags);
		logdk("complete urb %p seq=0x%x/%d\n", urb, urbpriv_seq(urbpriv), status);

		/* Copy urb buffer */
		if (usb_pipeisoc(urb->pipe)) {
			urb->error_count = 0;
			piso = (iso_frame_t *)((urb_t *)arg)->buffer;
			for (i = 0; i < urb->number_of_packets; i++, piso ++) {
				if (get_user(iso.offset, (uint32_t __user *)&piso->offset)
				    || get_user(iso.length, (uint32_t __user *)&piso->length)
				    || get_user(iso.status, (int32_t __user *)&piso->status)) {
					urbpriv_unlock_unref(urbpriv);
					return -EFAULT;
				}
				if (iso.offset + iso.length > urb->transfer_buffer_length
				    || iso.length > urb->iso_frame_desc[i].length) {
					urbpriv_unlock_unref(urbpriv);
					return -ERANGE;
				}
				if (usb_pipein(urb->pipe))
					urb->iso_frame_desc[i].offset = iso.offset;
				urb->iso_frame_desc[i].actual_length = iso.length;
				urb->iso_frame_desc[i].status = iso.status;
				if (iso.status != 0)
					urb->error_count++;
			}
		}
		size = urbhdr.size - (sizeof(urb_t) + offset);
		if (size > urb->transfer_buffer_length)
			size = urb->transfer_buffer_length;
		if (size) {
			if (urb->num_sgs) {
				for_each_sg(urb->sg, sg, urb->num_sgs, i) {
					sz = min(size, sg->length);
					if (copy_from_user(sg_virt(sg), ((urb_t *)arg)->buffer + offset, sz)) {
						urbpriv_unlock_unref(urbpriv);
						return -EFAULT;
					}
					offset += sz;
					size -= sz;
					if (!size)
						break;
				}
			} else {
				if (copy_from_user(urb->transfer_buffer, ((urb_t *)arg)->buffer + offset, size)) {
					urbpriv_unlock_unref(urbpriv);
					return -EFAULT;
				}
			}
		}

		hcd = urbpriv_port(urbpriv) < FTHC_HS_MAX_PORTS ? hcd_instance->hcd_hs : hcd_instance->hcd_ss;
		urbpriv_unlink(urbpriv);
		urbpriv_unset_submitted(urbpriv);
		urbpriv_unlock_unref(urbpriv);

		spin_lock_irqsave(&hcd_instance->lock, flags);
#if LINUX_VERSION_CODE != KERNEL_VERSION(2,6,18)
		usb_hcd_unlink_urb_from_ep(hcd, urb);
#endif
		spin_unlock_irqrestore(&hcd_instance->lock, flags);
#if LINUX_VERSION_CODE != KERNEL_VERSION(2,6,18)
		usb_hcd_giveback_urb(hcd, urb, status);
#else
		usb_hcd_giveback_urb(hcd, urb, NULL);
#endif
		return 0;

	case IOCTL_GET_HCD_DEV_ADDR:
		logdk("ioctl: get hcd device address\n");

		if (copy_from_user(&hda, arg, sizeof(hcd_dev_addr_t)))
			return -EFAULT;

		hda.busnum = 0;
		hda.devnum = 0;
		spin_lock_irqsave(&hcd_instance->lock, flags);
		if (hcd_instance->ports[hda.port.port_number].usbdev != NULL) {
			hda.busnum = hcd_instance->ports[hda.port.port_number].usbdev->bus->busnum;
			hda.devnum = hcd_instance->ports[hda.port.port_number].usbdev->devnum;
		}
		spin_unlock_irqrestore(&hcd_instance->lock, flags);

		if (copy_to_user(arg, &hda, sizeof(hda)))
			return -EFAULT;
		return 0;

	default:
		logdk("unknown ioctl=%x\n", ioctl_num);
	}
	return -ENOSYS;
}

static int link_open(struct inode *inode, struct file *file) {
	int rc = fthc_refcnt_inc();
	if (rc == -EMLINK)
		logdk("only one connection is supported\n");
	else if (rc)
		logdk("fail to open link device: %d\n", rc);
	else
		logdk("link device opened\n");
	return 0;
}

static int link_close(struct inode *inode, struct file *file) {
	int np;
	hcd_port_t port;
	logdk("link device closed\n");
	/* disable all ports */
	for (np = 0; np < FTHC_MAX_PORTS; np++) {
		port.port_number = np;
		fthc_port_disable(&port);
	}
	usb_hcd_poll_rh_status(hcd_instance->hcd_hs);
#ifdef USB_DT_SS_HUB
	usb_hcd_poll_rh_status(hcd_instance->hcd_ss);
#endif
	fthc_refcnt_dec();
	return 0;
}

/**
 * Declare link device
 */
static struct file_operations linkdev_fops = {
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 35)
	.ioctl        = link_ioctl,
#else
	.unlocked_ioctl    = link_ioctl,
	.compat_ioctl      = link_ioctl,
#endif
	.poll = link_poll,
	.open = link_open,
	.release = link_close,
};

static struct miscdevice linkdev = {
	.minor = MISC_DYNAMIC_MINOR,
	.name = LINK_DEVICE_NAME,
	.fops = &linkdev_fops,
};


/**
 * Function to create and destroy link device
 */
int link_create(void) {
	urbpriv_list_init();
	return misc_register(&linkdev);
}

void link_destroy(void) {
	misc_deregister(&linkdev);
}


