/*
 * Copyright (c) 2024 FabulaTech.
 * All rights reserved.
 * Permission to use this software as a part of FabulaTech solution only is
 * granted.
 * http://www.fabulatech.com
 */

#include <linux/module.h>

#define DEBUG 

#include <linux/spinlock.h>
#include <linux/platform_device.h>
#include <linux/usb.h>
#include <linux/device.h>

#include "ftrelease.h"
#include "ftdefs.h"

/* The only instance of virtual hcd */
static struct hcdpriv fthcd;
struct hcdpriv *hcd_instance = &fthcd;

#define PORT_C_MASK \
	((USB_PORT_STAT_C_CONNECTION \
	| USB_PORT_STAT_C_ENABLE \
	| USB_PORT_STAT_C_SUSPEND \
	| USB_PORT_STAT_C_OVERCURRENT \
	| USB_PORT_STAT_C_RESET) << 16)

static const char *hub_port_stat_desc[] = {
	"CONNECTION",		/*0*/
	"ENABLE",		/*1*/
	"SUSPEND",		/*2*/
	"OVER_CURRENT",		/*3*/
	"RESET",		/*4*/
	"R5", "R6", "R7",
	"POWER",		/*8*/
	"LOWSPEED",		/*9*/
	"HIGHSPEED",		/*10*/
	"PORT_TEST",		/*11*/
	"INDICATOR",		/*12*/
	"R13", "R14", "R15",
	"C_CONNECTION",		/*16*/
	"C_ENABLE",		/*17*/
	"C_SUSPEND",		/*18*/
	"C_OVER_CURRENT",	/*19*/
	"C_RESET",		/*20*/
	"R21", "R22", "R23", "R24", "R25", "R26", "R27", "R28", "R29", "R30", "R31",
};

/**
 * Host controller initialization
 */
static int fthc_start(struct usb_hcd *hcd) {
	int np;

	logdk("fthc %s start\n", hcd == hcd_instance->hcd_hs ? "HS" : "SS");
	/* Initialize hcd */
	hcd->power_budget = 0;
	hcd->state = HC_STATE_RUNNING;
	hcd->uses_new_polling = 1;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,26)
	if (usb_hcd_is_primary_hcd(hcd))
		hcd->has_tt = 1;
#endif
	
	for (np = 0; np < FTHC_MAX_PORTS; np++) {
		hcd_instance->ports[np].pending = 0;
	}

	/* Create link to usermode daemon */
	if (hcd == hcd_instance->hcd_hs)
		link_create();

	return 0;
}

/**
 * Host controller cleanup
 */
static void fthc_stop(struct usb_hcd *hcd) {
	logdk("fthc %s stop\n", hcd == hcd_instance->hcd_hs ? "HS" : "SS");
	/* Destroy usermode link */
	if (hcd == hcd_instance->hcd_hs)
		link_destroy();
}

static int fthc_reset(struct usb_hcd *hcd) {
	logdk("fthc %s reset\n", usb_hcd_is_primary_hcd(hcd) ? "HS" : "SS");
#ifdef USB_DT_SS_HUB
	if (usb_hcd_is_primary_hcd(hcd)) {
		hcd->speed = HCD_USB2;
		hcd->self.root_hub->speed = USB_SPEED_HIGH;
	} else {
		hcd->speed = HCD_USB3;
		hcd->self.root_hub->speed = USB_SPEED_SUPER;
	}
#else
	hcd->self.root_hub->speed = USB_SPEED_HIGH;
#endif
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,12,0)
	hcd->self.sg_tablesize = ~0;
	hcd->self.no_sg_constraint = 1;
#endif
	return 0;
}

/**
 * Enable the port on hcd
 */
int fthc_port_enable(hcd_port_t *port) {
	int np;
	unsigned long flags;

	spin_lock_irqsave(&hcd_instance->lock, flags);
	if (port->speed < FTHC_SPEED_SUPER) {
		if (port->port_number < 0) {
			for (np = 0; np < FTHC_HS_MAX_PORTS; np++) {
				if ((hcd_instance->ports[np].status
					& (1 << USB_PORT_STAT_CONNECTION)) == 0
					&& hcd_instance->ports[np].speed == 0
					&& !hcd_instance->ports[np].pending
					&& hcd_instance->ports[np].usbdev == NULL) {
					port->port_number = np;
					break;
				}
			}
		} else
			np = port->port_number;
		if (np >= FTHC_HS_MAX_PORTS)
			goto no_port;
	} else {
		if (port->port_number < 0) {
			for (np = FTHC_HS_MAX_PORTS; np < FTHC_MAX_PORTS; np++) {
				if ((hcd_instance->ports[np].status
					& (1 << USB_PORT_STAT_CONNECTION)) == 0
					&& hcd_instance->ports[np].speed == 0
					&& !hcd_instance->ports[np].pending
					&& hcd_instance->ports[np].usbdev == NULL) {
					port->port_number = np;
					break;
				}
			}
		} else
			np = port->port_number;
		if (np >= FTHC_MAX_PORTS)
			goto no_port;
	}
	hcd_instance->ports[np].speed = port->speed;
	hcd_instance->ports[np].status |= USB_PORT_STAT_ENABLE;
	hcd_instance->ports[np].status |= USB_PORT_STAT_CONNECTION
		| (1 << USB_PORT_FEAT_C_CONNECTION);
	switch (port->speed) {
	case FTHC_SPEED_LOW:
		hcd_instance->ports[np].status |= USB_PORT_STAT_LOW_SPEED;
		break;
	case FTHC_SPEED_HIGH:
		hcd_instance->ports[np].status |= USB_PORT_STAT_HIGH_SPEED;
		break;
	case FTHC_SPEED_FULL:
	case FTHC_SPEED_SUPER:
	default:
		break;
	}
	hcd_instance->ports[np].usbdev = NULL;
	hcd_instance->ports[np].pending = 1;
	spin_unlock_irqrestore(&hcd_instance->lock, flags);

	switch (port->speed) {
	case FTHC_SPEED_LOW:
		printk(KERN_INFO "new low speed device at port %d\n", np);
		break;
	case FTHC_SPEED_HIGH:
		printk(KERN_INFO "new high speed device at port %d\n", np);
		break;
	case FTHC_SPEED_FULL:
		printk(KERN_INFO "new full speed device at port %d\n", np);
		break;
	case FTHC_SPEED_SUPER:
	default:
		printk(KERN_INFO "new super speed device at port %d\n", np);
	}

	return 0;

no_port:
	spin_unlock_irqrestore(&hcd_instance->lock, flags);
	printk(KERN_ERR "no such port: %d\n", np);
	return -EINVAL;
}

/**
 * Disable the port on hcd
 */
void fthc_port_disable(hcd_port_t *port) {
	int np;
	unsigned long flags;

	spin_lock_irqsave(&hcd_instance->lock, flags);
	if (port->port_number < 0 || port->port_number >= FTHC_MAX_PORTS) {
		spin_unlock_irqrestore(&hcd_instance->lock, flags);
		printk(KERN_ERR "no such port: %d\n", port->port_number);
		return;
	}
	np = port->port_number;
	hcd_instance->ports[np].status &= ~(USB_PORT_STAT_CONNECTION | USB_SPEED_HIGH | USB_SPEED_LOW);
	hcd_instance->ports[np].status |= (1 << USB_PORT_FEAT_C_CONNECTION);
	hcd_instance->ports[np].speed = 0;
	hcd_instance->ports[np].usbdev = NULL;
	hcd_instance->ports[np].pending = 0;
	spin_unlock_irqrestore(&hcd_instance->lock, flags);

	logdk("port %d disconnected\n", np);
}

/** 
 * Called when urb is enqueued (submitted)
 */
#if LINUX_VERSION_CODE != KERNEL_VERSION(2,6,18)
static int fthc_urb_enqueue (struct usb_hcd *hcd, struct urb *urb, gfp_t mem_flags)
#else
static int fthc_urb_enqueue (struct usb_hcd *hcd,
		struct usb_host_endpoint *ep,
		struct urb *urb,
		gfp_t mem_flags)
#endif
{
	unsigned long flags;
	int i, np, rc;
	struct urbpriv *urbpriv;

	logdk("enqueueing urb %p\n", urb);

	/* Preallocate usbpriv structure when spinlock is not held */
	urbpriv = urbpriv_alloc(mem_flags);
	if (urbpriv == NULL)
		return -EILSEQ;

	spin_lock_irqsave(&hcd_instance->lock, flags);

	if (urb->status != -EINPROGRESS) {
		rc = urb->status;
		spin_unlock_irqrestore(&hcd_instance->lock, flags);
		urbpriv_unref(urbpriv);
		printk(KERN_ERR "urb %p is already unlinked with status %d\n", urb, rc);
		return rc;
	}

#if LINUX_VERSION_CODE != KERNEL_VERSION(2,6,18)
	/* Try to link urb to the endpoint */
	rc = usb_hcd_link_urb_to_ep(hcd, urb);
	if (rc) {
		spin_unlock_irqrestore(&hcd_instance->lock, flags);
		urbpriv_unref(urbpriv);
		printk(KERN_ERR "cannot link urb %p: %d\n", urb, rc);
		return rc;
	}
#endif

	/* 
	 * Check if urb is linked to any device. 
	 */
	if (usb_pipedevice(urb->pipe) == 0) {
		/* Expect control request: setAddress or getDescriptor */
		if (usb_pipetype(urb->pipe) == PIPE_CONTROL) {
			struct usb_ctrlrequest *usbctlreq = (struct usb_ctrlrequest *) urb->setup_packet;
			for (i = 0, np = -1; i < FTHC_MAX_PORTS; i++) {
				if (hcd_instance->ports[i].pending
					&& (hcd_instance->ports[i].usbdev == NULL
						|| hcd_instance->ports[i].usbdev == urb->dev)) {
					np = i;
					break;
				}
			}
			if (np == -1) {
				printk(KERN_ERR "no pending ports found\n");
				rc = -EINVAL;
				goto unlink;
			} else {
				logdk("port %d device %p\n", i, urb->dev);
			}


			logdk("early control req:%x val:%x\n", usbctlreq->bRequest, usbctlreq->wValue);

			switch (usbctlreq->bRequest) {
			case USB_REQ_SET_ADDRESS:
				logdk("set address request\n");
				/* associate device with the hcd port */
				hcd_instance->ports[np].usbdev = urb->dev;
				if (urb->status == -EINPROGRESS) {
					urb->status = 0;
				}
				goto giveback;
			case USB_REQ_GET_DESCRIPTOR:
				hcd_instance->ports[np].usbdev = urb->dev;
				break;
			}
		} else { /* Is it possible? */
			/* TODO: print more details */
			printk(KERN_ERR "unexpected transfer for unintialized device\n");
			rc = -EINVAL;
			goto unlink;
		}
	}

	for (i = 0, np = -1; i < FTHC_MAX_PORTS; i++) {
		if (hcd_instance->ports[i].usbdev == urb->dev) {
			np = i;
			break;
		}
	}
	if (np < 0) {
		printk(KERN_ERR "device %p without port\n", urb->dev);
		rc = -EINVAL;
		goto unlink;
	}

	/* Send to user space */
	link_urb_enqueue(urb, urbpriv, np);

	spin_unlock_irqrestore(&hcd_instance->lock, flags);
	urbpriv_unref(urbpriv);
	return 0;

unlink:
#if LINUX_VERSION_CODE != KERNEL_VERSION(2,6,18)
	usb_hcd_unlink_urb_from_ep(hcd, urb);
#endif
	spin_unlock_irqrestore(&hcd_instance->lock, flags);
	urbpriv_unref(urbpriv);
	return rc;

giveback:
#if LINUX_VERSION_CODE != KERNEL_VERSION(2,6,18)
	usb_hcd_unlink_urb_from_ep(hcd, urb);
#endif
	spin_unlock_irqrestore(&hcd_instance->lock, flags);
#if LINUX_VERSION_CODE != KERNEL_VERSION(2,6,18)
	usb_hcd_giveback_urb(hcd, urb, urb->status);
#else
	usb_hcd_giveback_urb(hcd, urb, NULL);
#endif
	urbpriv_unref(urbpriv);
	return 0;
}

/**
 * Called when urb is removed from the queue (cancelled)
 */
#if LINUX_VERSION_CODE != KERNEL_VERSION(2,6,18)
static int fthc_urb_dequeue (struct usb_hcd *hcd, struct urb *urb, int status)
#else
static int fthc_urb_dequeue (struct usb_hcd *hcd, struct urb *urb)
#endif
{
	unsigned long flags;
	int rc;

	logdk("dequeueing urb %p\n", urb);

	rc = link_urb_dequeue(urb);
	if (rc)
		return rc;

	spin_lock_irqsave(&hcd_instance->lock, flags);
#if LINUX_VERSION_CODE != KERNEL_VERSION(2,6,18)
	rc = usb_hcd_check_unlink_urb(hcd, urb, status);
	if (rc) {
		spin_unlock_irqrestore(&hcd_instance->lock, flags);
		return rc;
	}
	usb_hcd_unlink_urb_from_ep(hcd, urb);
#endif
	spin_unlock_irqrestore(&hcd_instance->lock, flags);
#if LINUX_VERSION_CODE != KERNEL_VERSION(2,6,18)
	usb_hcd_giveback_urb(hcd, urb, status);
#else
	usb_hcd_giveback_urb(hcd, urb, NULL);
#endif
	return 0;
}

/**
 * Return hub status information
 */
static int fthc_hub_status (struct usb_hcd *hcd, char *buf)
{
	int rc, i, n, m, changed;
	unsigned long flags;

	logdk("hub %s status request\n", hcd == hcd_instance->hcd_hs ? "HS" : "SS");

	changed = 0;
	rc = DIV_ROUND_UP((hcd == hcd_instance->hcd_hs ? FTHC_HS_MAX_PORTS : FTHC_SS_MAX_PORTS) + 1, 8);
	memset(buf, 0, rc);

	spin_lock_irqsave(&hcd_instance->lock, flags);

	if (!test_bit(HCD_FLAG_HW_ACCESSIBLE, &hcd->flags)) {
		goto done;
	}

	n = hcd == hcd_instance->hcd_hs ? 0 : FTHC_HS_MAX_PORTS;
	m = hcd == hcd_instance->hcd_hs ? FTHC_HS_MAX_PORTS : FTHC_SS_MAX_PORTS;
	for (i = 0; i < m; i++) {
		if ((hcd_instance->ports[i + n].status & PORT_C_MASK) != 0) {
			buf[(i + 1) / 8] |= 1 << (i + 1) % 8;
			changed = 1;
		}
	}

	if ((hcd->state == HC_STATE_SUSPENDED) && (changed == 1))
		usb_hcd_resume_root_hub(hcd);

done:
	spin_unlock_irqrestore(&hcd_instance->lock, flags);
	return changed ? rc : 0;
}

/* Fill hub descriptor */
static inline void hub_descriptor (struct usb_hub_descriptor *desc) {
	int cnt = (FTHC_HS_MAX_PORTS + 1 + 7) / 8;
	memset (desc, 0, sizeof *desc);
	desc->bDescriptorType = 0x29;
	desc->bDescLength = 7 + 2 * cnt;
	desc->wHubCharacteristics = cpu_to_le16(0x0001);
	desc->bNbrPorts = FTHC_HS_MAX_PORTS;
#ifdef USB_DT_SS_HUB
	memset(&desc->u.hs.DeviceRemovable[0], 0xff, cnt);
	memset(&desc->u.hs.PortPwrCtrlMask[0], 0xff, cnt);
#else
	memset(&desc->DeviceRemovable[0], 0xff, cnt);
	memset(&desc->PortPwrCtrlMask[0], 0xff, cnt);
#endif
}

#ifdef USB_DT_SS_HUB
static inline void ss_hub_descriptor(struct usb_hub_descriptor *desc) {
	memset(desc, 0, sizeof *desc);
	desc->bDescriptorType = USB_DT_SS_HUB;
	desc->bDescLength = USB_DT_SS_HUB_SIZE;
	desc->wHubCharacteristics = cpu_to_le16(0x0001);
	desc->bNbrPorts = FTHC_SS_MAX_PORTS;
	desc->u.ss.bHubHdrDecLat = 0x04; /* Worst case: 0.4 micro sec*/
	desc->u.ss.DeviceRemovable = 0xffff;
}

static struct {
	struct usb_bos_descriptor bos;
	struct usb_ss_cap_descriptor ss_cap;
} __packed usb3_bos_desc = {

	.bos = {
		.bLength		= USB_DT_BOS_SIZE,
		.bDescriptorType	= USB_DT_BOS,
		.wTotalLength		= cpu_to_le16(sizeof(usb3_bos_desc)),
		.bNumDeviceCaps		= 1,
	},
	.ss_cap = {
		.bLength		= USB_DT_USB_SS_CAP_SIZE,
		.bDescriptorType	= USB_DT_DEVICE_CAPABILITY,
		.bDevCapabilityType	= USB_SS_CAP_TYPE,
		.wSpeedSupported	= cpu_to_le16(USB_5GBPS_OPERATION),
		.bFunctionalitySupport	= ilog2(USB_5GBPS_OPERATION),
	},
};
#endif

/**
 * 
 */
static int fthc_hub_control(struct usb_hcd *hcd, uint16_t wRequest,
		uint16_t wValue, uint16_t wIndex,
		char *buf, uint16_t len)
{
	int i, rc, port, changed;
	unsigned long flags;
	struct urbpriv *urbpriv;

	logdk("hub %s control req:%04x v:%04x i:%04x l:%d\n",
	    hcd == hcd_instance->hcd_hs ? "HS" : "SS",
	    wRequest, wValue, wIndex, len);

	if (wRequest == SetPortFeature && wValue == USB_PORT_FEAT_RESET) {
		/* Preallocate usbpriv structure when spinlock is not held */
		urbpriv = urbpriv_alloc(GFP_ATOMIC);
		if (urbpriv == NULL)
			return -EILSEQ;
	} else
		urbpriv = NULL;

	spin_lock_irqsave(&hcd_instance->lock, flags);

	switch (wRequest) {
		case ClearPortFeature:
		case GetPortStatus:
		case SetPortFeature:
			port = (wIndex & 0x00ff) - 1;
			if (hcd == hcd_instance->hcd_hs) {
				if (port < 0 || port >= FTHC_HS_MAX_PORTS) { /* requested invalid port */
					rc = -EPIPE;
					goto done;
				}
			} else {
				port += FTHC_HS_MAX_PORTS;
				if (port < FTHC_HS_MAX_PORTS || port >= FTHC_MAX_PORTS) { /* requested invalid port */
					rc = -EPIPE;
					goto done;
				}
			}
			break;
		default:
			port = 0;
	}

	if (!test_bit(HCD_FLAG_HW_ACCESSIBLE, &hcd->flags)) {
		spin_unlock_irqrestore(&hcd_instance->lock, flags);
		if (urbpriv != NULL)
			urbpriv_unref(urbpriv);
		return -ETIMEDOUT;
	}

	rc = 0;
	switch (wRequest) {
		case ClearHubFeature:
			break;
		case ClearPortFeature:
			logdk("clear port feature: %s\n", hub_port_stat_desc[wValue]);
			switch (wValue) {
#ifdef USB_DT_SS_HUB
			case USB_PORT_FEAT_POWER:
				hcd_instance->ports[port].status &= USB_SS_PORT_STAT_POWER;
				hcd_instance->ports[port].status &= USB_PORT_STAT_POWER;
				break;
#endif
			default:
				hcd_instance->ports[port].status &= ~(1 << wValue);
			}
			break;
		case GetHubDescriptor:
			if (hcd == hcd_instance->hcd_hs)
				hub_descriptor((struct usb_hub_descriptor *) buf);
#ifdef USB_DT_SS_HUB
			else
				ss_hub_descriptor((struct usb_hub_descriptor *) buf);
#endif
			break;
#ifdef USB_DT_SS_HUB
		case DeviceRequest | USB_REQ_GET_DESCRIPTOR:
			memcpy(buf, &usb3_bos_desc,
			    len < sizeof(usb3_bos_desc) ? len : sizeof(usb3_bos_desc));
			rc = sizeof(usb3_bos_desc);
			break;
#endif
		case GetHubStatus:
			*(__le32 *)buf = cpu_to_le32(0);
			break;
		case GetPortStatus:
			logdk("get port status: port=%d, status=%08x\n", port, hcd_instance->ports[port].status);
			if ((hcd_instance->ports[port].status & USB_PORT_STAT_RESET)
			    && time_after_eq (jiffies, hcd_instance->ports[port].re_timeout)) {
				hcd_instance->ports[port].status |= (USB_PORT_STAT_C_RESET << 16);
				hcd_instance->ports[port].status &= ~USB_PORT_STAT_RESET;
				hcd_instance->ports[port].status |= USB_PORT_STAT_ENABLE;
				hcd_instance->ports[port].usbdev = NULL;
				if (hcd_instance->ports[port].speed == FTHC_SPEED_HIGH) {
					logdk("reset high speed device complete\n");
					hcd_instance->ports[port].status |= USB_PORT_STAT_HIGH_SPEED;
				} else if (hcd_instance->ports[port].speed == FTHC_SPEED_LOW) {
					logdk("reset low speed device complete\n");
					hcd_instance->ports[port].status |= USB_PORT_STAT_LOW_SPEED;
				} else {
					logdk("reset full speed device complete\n");
				}
			}
			((__le16 *) buf)[0] = cpu_to_le16 (hcd_instance->ports[port].status);
			((__le16 *) buf)[1] = cpu_to_le16 (hcd_instance->ports[port].status >> 16);
			break;
		case SetHubFeature:
			rc = -EPIPE;
			break;
		case SetPortFeature:
			logdk("set port %d feature: %s\n", port, hub_port_stat_desc[wValue]);
			switch (wValue) {
				case USB_PORT_FEAT_POWER:
					if (hcd == hcd_instance->hcd_hs)
						hcd_instance->ports[port].status |= USB_PORT_STAT_POWER;
#ifdef USB_DT_SS_HUB
					else
						hcd_instance->ports[port].status |= USB_SS_PORT_STAT_POWER;
#endif
					break;
				case USB_PORT_FEAT_RESET:
					/* disable port */
					if (hcd == hcd_instance->hcd_hs) {
						hcd_instance->ports[port].status &= ~(USB_PORT_STAT_ENABLE
								| USB_PORT_STAT_LOW_SPEED
								| USB_PORT_STAT_HIGH_SPEED);
					}
#ifdef USB_DT_SS_HUB
					else {
						hcd_instance->ports[port].status = (USB_SS_PORT_STAT_POWER
							 | USB_PORT_STAT_CONNECTION
							 | USB_PORT_STAT_RESET);
					}
#endif

					/* set timeout until reset is done */
					hcd_instance->ports[port].status |= USB_PORT_STAT_RESET;
					hcd_instance->ports[port].re_timeout = jiffies + msecs_to_jiffies(50);
					
					logdk("reset started, port=%d\n", port);

					link_urb_special(urbpriv, FUNC_RESET, port);
					break;
				default:
					if (hcd == hcd_instance->hcd_hs) {
						if ((hcd_instance->ports[port].status & USB_PORT_STAT_POWER) != 0)
							hcd_instance->ports[port].status |= (1 << wValue);
					}
#ifdef USB_DT_SS_HUB
					else {
						if ((hcd_instance->ports[port].status & USB_SS_PORT_STAT_POWER) != 0)
							hcd_instance->ports[port].status |= (1 << wValue);
					}
#endif
					break;
			}
			break;
		default:
			rc = -EPIPE;
			break;
	}

done:
	changed = 0;
	for (i = 0; i < FTHC_MAX_PORTS; i++) {
		if ((hcd_instance->ports[i].status & PORT_C_MASK) != 0) {
			changed = 1;
			break;
		}
	}

	spin_unlock_irqrestore(&hcd_instance->lock, flags);
	if (urbpriv != NULL)
		urbpriv_unref(urbpriv);

	if (changed)
		usb_hcd_poll_rh_status (hcd);

	return rc;
}

#ifdef CONFIG_PM
static int fthc_hub_suspend(struct usb_hcd *hcd)
{
	unsigned long flags;
	logdk("fthc %s hub suspend\n", hcd == hcd_instance->hcd_hs ? "HS" : "SS");
	spin_lock_irqsave(&hcd_instance->lock, flags);
	hcd->state = HC_STATE_SUSPENDED;
	spin_unlock_irqrestore(&hcd_instance->lock, flags);
	return 0;
}

static int fthc_hub_resume(struct usb_hcd *hcd)
{
	unsigned long flags;
	int rc;
	logdk("fthc %s hub resume\n", hcd == hcd_instance->hcd_hs ? "HS" : "SS");
	spin_lock_irqsave(&hcd_instance->lock, flags);
	if (!test_bit(HCD_FLAG_HW_ACCESSIBLE, &hcd->flags)) {
		logdk("hub isn't running\n");
		rc = -ENODEV;
	} else {
		hcd->state = HC_STATE_RUNNING;
		rc = 0;
	}
	spin_unlock_irqrestore(&hcd_instance->lock, flags);
	return rc;
}
#else
#define fthc_hub_suspend      NULL
#define fthc_hub_resume       NULL
#endif

static int fthc_get_frame(struct usb_hcd *hcd) {
#if LINUX_VERSION_CODE < KERNEL_VERSION(5,0,0)
	struct timeval tv;
	do_gettimeofday(&tv);
	return (tv.tv_usec / 1000);
#else
	struct timespec64 ts64;
	ktime_get_real_ts64(&ts64);
	return (ts64.tv_nsec / 1000000);
#endif
}

const struct hc_driver fthc_driver = {
	.description = (char *) FT_DRIVER_NAME,
	.product_desc = (char *) FT_DRIVER_DESCRIPTION,

	.hcd_priv_size = sizeof(struct hcdpriv),

#ifdef USB_DT_SS_HUB
	.flags = HCD_USB3 | HCD_SHARED,
#else
	.flags = HCD_USB2,
#endif

	.start = fthc_start,
	.stop = fthc_stop,
	.reset = fthc_reset,

	.urb_enqueue = fthc_urb_enqueue,
	.urb_dequeue = fthc_urb_dequeue,

	.get_frame_number = fthc_get_frame,

	.hub_status_data = fthc_hub_status,
	.hub_control = fthc_hub_control,
	.bus_suspend = fthc_hub_suspend,
	.bus_resume = fthc_hub_resume,
};
