#!/bin/sh

# This script integrates FabuleTech library to FreeRDP
#

#set -e

LIB_DIR="/opt/ftplugins/lib"
LIB_NAME_10="ftrdpbr.so"
LIB_NAME_11="ftrdpbr-client.so"
LIB_NAME_12="libftrdpbr-client.so"
LIB_NAMES="$LIB_NAME_10 $LIB_NAME_11 $LIB_NAME_12"

LINK_COUNT=0
NO_ARCH="unknown"
ARCH=$NO_ARCH

usage () {
echo "\
Usage: $(basename $0) -i|-u,
where:
        -i      - Install FreeRDP support
        -u      - Uninstall FreeRDP support
        -h      - Show help"
}

get_arch () {
	if [ "x$(uname -m | grep 'x86_64')" != "x" ]; then
		arch="x86_64"
	elif [ "x$(uname -m | grep 'i686')" != "x" ]; then
		arch=="i386"
	elif [ "x$(uname -m | grep 'arm')" != "x" ]; then
		arch="arm"
	elif [ "x$(uname -m | grep 'aarch64')" != "x" ]; then
		arch="aarch64"
	elif [ "x$(uname -m | grep 'mips64')" != "x" ]; then
		arch="mips64el"
	else
		arch=$NO_ARCH
	fi
	echo $arch
}

gnu_suffixes () {
	p1="gnu"
	p2="gnueabi"
	p3="gnueabihf"
	p4="gnuabi64"
	echo $p1 $p2 $p3 $p4
}

rdp_dirs () {
	p1="/usr/lib/freerdp"
	p2="/usr/local/lib/freerdp"
	p3="/usr/lib64/freerdp"
	p4="/usr/local/lib64/freerdp"
	echo $p1 $p2 $p3 $p4

	if [ "x$ARCH" != "x$NO_ARCH" ]
	then
		suffixes=$(gnu_suffixes)
		for suf in $suffixes
		do
			p1="/usr/lib/$ARCH-linux-$suf/freerdp"
			p2="/usr/local/lib/$ARCH-linux-$suf/freerdp"
			p3="/usr/lib64/$ARCH-linux-$suf/freerdp"
			p4="/usr/local/lib64/$ARCH-linux-$suf/freerdp"
			echo $p1 $p2 $p3 $p4
		done
	fi
}

rdp1_dirs () {
	p1="/usr/lib/libfreerdp-1_0"
	echo "$(rdp_dirs) $p1" 
}

rdp2_dirs () {
	dirs=$(rdp_dirs)
	for dir in $dirs
	do
		echo "$dir"2
	done
}

rdp3_dirs () {
	dirs=$(rdp_dirs)
	for dir in $dirs
	do
		echo "$dir"3
	done
}

unlink_plugin () {
	dir=$1
	name=$2

	path="$dir/$name"
	if [ -h $path -o -f $path ]
	then
		rm $path
		if [ $? = 0 ]
		then
			echo "removed: $path"
		else
			echo "WARNING: unable to remove: $path"
		fi
	fi
}

link_plugin () {
	dir=$1
	name=$2

	if [ -d $dir ]
	then
		path="$dir/$name"
		lib="$LIB_DIR/$name"
		ln -s $lib $path
		if [ $? = 0 ]
		then
			echo "linked: $path"
			LINK_COUNT=$(($LINK_COUNT + 1))
		else
			echo "WARNING: unable to create link: $path"
		fi
	fi
}

remove_dirs() {
	dirs="$(rdp2_dirs) $(rdp3_dirs)"
	for dir in $dirs
	do
		if [ -d $dir ]; then
			rmdir --ignore-fail-on-non-empty $dir
		fi
	done
}

unlink_all () {
	dirs="$(rdp1_dirs) $(rdp2_dirs) $(rdp3_dirs)"
	for dir in $dirs
	do
		for name in $LIB_NAMES
		do
			unlink_plugin $dir $name
		done
	done
}

create_dirs () {
	major=$1
	
	dirs="/usr/lib /usr/lib64 /usr/local/lib"
	for dir1 in $dirs
	do
		dir2="$dir1/freerdp$major"
		if [ -d  $dir1 -a ! -d $dir2  ]
		then
			mkdir $dir2
			if [ $? != 0 ]
			then
				echo "WARNING: unable to cteate directory: $dir2"
			fi
		fi
	done
	if [ "x$ARCH" != "x$NO_ARCH" ]
	then
		suffixes=$(gnu_suffixes)
		for suf in $suffixes
		do
			dirs="/usr/lib/$ARCH-linux-$suf /usr/local/lib/$ARCH-linux-$suf"
			for dir1 in $dirs
			do
				dir2="$dir1/freerdp$major"
				if [ -d  $dir1 -a ! -d $dir2  ]
				then
					mkdir $dir2
					if [ $? != 0 ]
					then
						echo "WARNING: unable to cteate directory: $dir2"
					fi
				fi
			done
		done
	fi
}

link_version () {
	major=$1
	minor=$2

	# For now, FreeRDP version 2.0.0 and above doesn't create directory for plugins.
	# The same goes for third party applications using libfreerdpXX.
	if [ $major -eq 2 -o  $major -eq 3 ]
	then
		create_dirs $major
	fi
	
	case $major in
	1)
		dirs=$(rdp1_dirs)
		for dir in $dirs
		do
			case $minor in
			0)
				link_plugin $dir $LIB_NAME_10
				;;
			1)
				link_plugin $dir $LIB_NAME_11
				;;
			*)
				link_plugin $dir $LIB_NAME_12
				;;
			esac
		done
		;;
	2)
		dirs=$(rdp2_dirs)
		for dir in $dirs
		do
			link_plugin $dir $LIB_NAME_12
		done
		;;
	3)
		dirs=$(rdp3_dirs)
		for dir in $dirs
		do
			link_plugin $dir $LIB_NAME_12
		done
		;;
	*)
		dirs="$(rdp1_dirs) $(rdp2_dirs) $(rdp3_dirs)"
		for dir in $dirs
		do
			for name in $LIB_NAMES
			do
				link_plugin $dir $name
			done
		done
		;;
	esac
}

link_all () {
	link_version 1 0
	link_version 1 1
	link_version 2 0
	link_version 3 0
}

ARCH=$(get_arch)
if [ "x$ARCH" = "x$NO_ARCH" ]
then
	echo "WARNING: unable to detect machine type"
fi

case $1 in
"-i")
	unlink_all
	link_all
	;;
"-u")
	unlink_all
	remove_dirs
	;;
"-h")
	usage
	;;
*)
	usage
	;;
esac

